<?php
/**
 * salary_calculate.php
 * Converts attendance aggregates into final salary figures
 */

function calculateSalaries(
    array $attendanceData,
    array $prefetch,
    array $context
): array {

    $branch      = $context['branch'];
    $month       = $context['month'];
    $year        = $context['year'];
    $calc        = $prefetch['calculation'];
    $employees   = $prefetch['employees'];

    $result = [];

    foreach ($attendanceData as $empId => $att) {

        if (!isset($employees[$empId])) {
            continue;
        }

        $emp = $employees[$empId];

        // ------------------------------------
        // BASE SALARY CALCULATION
        // ------------------------------------
        $perDaySalary = round($emp['salary'] / 26);
        $earnedSalary = round($perDaySalary * $att['b_a']);

        // ------------------------------------
        // BASIC & HRA (Branch-specific rules)
        // ------------------------------------
        if (in_array($branch, [4, 14, 41, 46, 47], true)) {
            $basic = $earnedSalary;
            $hra   = 0;
        } elseif ($branch === 21) {
            $basic = $earnedSalary;
            $hra   = round(($basic * 8.33) / 100, 2);
        } else {
            $basic = round(($earnedSalary * $calc['basic']) / 100);
            $hra   = round(($earnedSalary * $calc['hra']) / 100);
        }

        // ------------------------------------
        // PF & ESIC
        // ------------------------------------
        $pf = ($emp['pf'] === 'y')
            ? round(($basic * $calc['pf']) / 100)
            : 0;

        $esic = ($emp['esi'] === 'y')
            ? ceil(($earnedSalary * $calc['esic']) / 100)
            : 0;

        // ------------------------------------
        // TAX (Professional Tax)
        // ------------------------------------
        if ($earnedSalary >= 30000) {
            $tax = 200;
        } elseif ($earnedSalary >= 14000) {
            $tax = 125;
        } else {
            $tax = 0;
        }

        // ------------------------------------
        // TOTALS
        // ------------------------------------
        $gross = $basic + $hra + $att['spc_allow'];
        $deduction = round(
            $att['maual_advance'] + $tax + $pf + $esic
        );
        $netPay = round($gross - $deduction);

        // ------------------------------------
        // CHEQUE LOGIC
        // ------------------------------------
        $cheque = (
            empty($emp['bank_ac_no']) ||
            strtoupper($emp['bank_ac_no']) === 'NA' ||
            $att['cheque'] === 'yes'
        ) ? 'yes' : '';

        // ------------------------------------
        // FINAL ROW
        // ------------------------------------
        $result[] = [
            'name'            => $emp['e_name'],
            'designation'     => $emp['designation'],
            'ac_no'           => $emp['bank_ac_no'],
            'emp_id'          => $empId,
            'month'           => $month,
            'year'            => $year,
            'attendance'      => $att['b_a'],
            't_at'            => $att['t_a'],
            'basic_pay'       => $basic,
            'hra'             => $hra,
            'convyance'       => 0,
            'medical'         => 0,
            'gross_pay'       => $gross,
            'pf'              => $pf,
            'esic'            => $esic,
            'advance'         => $att['maual_advance'],
            'tax'             => $tax,
            'spc_allow'       => $att['spc_allow'],
            'deduction'       => $deduction,
            'net_pay'         => $netPay,
            'block'           => $att['block'],
            'cheque'          => $cheque,
            'branch'          => $branch,
            'payment_status'  => 0,
            'details'         => $att['details']
        ];
    }

    return $result;
}
