<?php
/**
 * salary_context.php
 * Resolves salary generation scope and enforces hard guards
 */

function loadSalaryContext(PDO $conn, int $branch): array
{
    // 1. Resolve target month/year (PREVIOUS MONTH)
    $dateObj = new DateTime('first day of last month');
    $month   = $dateObj->format('m'); // 01–12
    $year    = $dateObj->format('Y');

    // 2. Guard: Attendance must exist
    $stmt = $conn->prepare(
        "SELECT COUNT(*) 
         FROM attendance 
         WHERE month = :month AND year = :year AND branch = :branch"
    );
    $stmt->execute([
        ':month'  => $month,
        ':year'   => $year,
        ':branch' => $branch
    ]);

    if ((int)$stmt->fetchColumn() === 0) {
        throw new Exception(
            "No attendance data found for {$dateObj->format('F Y')}. Salary generation aborted."
        );
    }

    // 3. Guard: Salary must NOT already exist
    $stmt = $conn->prepare(
        "SELECT COUNT(*) 
         FROM emp_salary 
         WHERE month = :month AND year = :year AND branch = :branch"
    );
    $stmt->execute([
        ':month'  => $month,
        ':year'   => $year,
        ':branch' => $branch
    ]);

    if ((int)$stmt->fetchColumn() > 0) {
        throw new Exception(
            "Salary already generated for {$dateObj->format('F Y')}. Delete it before regenerating."
        );
    }

    // 4. Return immutable context
    return [
        'month'        => $month,
        'year'         => $year,
        'branch'       => $branch,
        'displayMonth' => $dateObj->format('F Y')
    ];
}
