<?php
/**
 * salary_prepare_attendance.php
 * Builds salary_attendance-equivalent aggregation from attendance table
 * LOGIC: 100% preserved from original working code
 */

function prepareAttendance(PDO $conn, array $context, array $prefetch): array
{
    $month  = $context['month'];
    $year   = $context['year'];
    $branch = $context['branch'];

    $employees = $prefetch['employees'];
    $sites     = $prefetch['sites'];

    // 1. Fetch attendance EXACTLY like original code
    $stmt = $conn->prepare(
        "SELECT *
         FROM attendance
         WHERE month = :month
           AND year  = :year
           AND branch = :branch"
    );
    $stmt->execute([
        ':month'  => $month,
        ':year'   => $year,
        ':branch' => $branch
    ]);

    $aggregated = [];

    while ($att = $stmt->fetch(PDO::FETCH_ASSOC)) {

        $empId = $att['emp_id'];

        // 2. ORIGINAL BEHAVIOR:
        // If employee missing → SKIP, DO NOT FAIL
        if (!isset($employees[$empId])) {
            continue;
        }

        $emp = $employees[$empId];

        // 3. Build attendance detail string (site-name + attendance)
        $siteName = $sites[$att['site']] ?? 'SITE';
        $attDetail = $siteName . '-' . $att['attendance'];

        // Initialize row if first encounter
        if (!isset($aggregated[$empId])) {
            $aggregated[$empId] = [
                'emp_id'          => $empId,
                'month'           => $month,
                'year'            => $year,
                't_a'             => 0,
                'b_a'             => 0,
                'o_a'             => 0,
                'block'           => $att['block'],
                'cheque'          => $att['check'],
                'branch'          => $branch,
                'maual_advance'   => 0,
                'spc_allow'       => 0,
                'details'         => '',
                'ot_enabled'      => !empty($emp['ot'])
            ];
        }

        $row = &$aggregated[$empId];

        // 4. Totals
        $row['t_a'] += (int)$att['attendance'];
        $row['maual_advance'] += (int)$att['advance_deduction'];
        $row['spc_allow'] += (int)$att['spc_allow'];
        $row['details'] .= ($row['details'] ? ', ' : '') . $attDetail;

        // 5. BASIC / OT LOGIC (UNCHANGED)
        if (!$row['ot_enabled']) {
            $row['b_a'] = min(26, $row['b_a'] + (int)$att['attendance']);
        } else {
            if ($att['only_ot'] === 'yes') {
                $row['o_a'] += (int)$att['attendance'];
            } else {
                if ($row['b_a'] < 26) {
                    $remaining = 26 - $row['b_a'];
                    if ($att['attendance'] <= $remaining) {
                        $row['b_a'] += (int)$att['attendance'];
                    } else {
                        $row['b_a'] = 26;
                        $row['o_a'] += (int)$att['attendance'] - $remaining;
                    }
                } else {
                    $row['o_a'] += (int)$att['attendance'];
                }
            }
        }
    }

    return $aggregated;
}
